package cn.royan.util.deobfuscator.map;

import cn.royan.MapStack;
import com.google.common.net.UrlEscapers;
import com.google.common.reflect.TypeToken;
import com.google.gson.*;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;

import java.io.*;
import java.lang.reflect.Type;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.nio.file.*;
import java.util.List;
import net.minecraft.class_155;

public class MappingProvider {

    private static final Logger LOGGER = MapStack.LOGGER;

    public static final String MINECRAFT_VERSION = class_155.field_29733;

    public static final String YARN_META = "https://meta.fabricmc.net/v2/versions/yarn/" + MINECRAFT_VERSION;
    public static final String YARN_MAPPING = "https://maven.fabricmc.net/net/fabricmc/yarn/";

    public static final String STORAGE_DIRECTORY = "./config/mappings/";
    public static final String MAPPINGS_PATH = "mappings/mappings.tiny";

    public static String getYarnVersion() throws IOException {
        URL url = URI.create(YARN_META).toURL();
        URLConnection request = url.openConnection();
        String content = new String(request.getInputStream().readAllBytes(), StandardCharsets.UTF_8);
        Gson gson = new Gson();
        Type listType = new TypeToken<List<JsonObject>>() {}.getType();
        List<JsonObject> list = gson.fromJson(content, listType);

        return MINECRAFT_VERSION + "+build." +
                list.stream()
                .mapToInt(obj -> obj.get("build").getAsInt())
                .max()
                .orElse(-1) ;
    }

    synchronized public static FileInputStream getYarnMappingStream(String yarnVersion) throws IOException
    {
        File mappingFile = new File(STORAGE_DIRECTORY + getMappingFileNameFull(yarnVersion));
        if (!(mappingFile.exists() && mappingFile.isFile()))
        {
            String mappingJar = String.format("%s.jar", getMappingFileName(yarnVersion));
            String mappingJarUrl = String.format("%s%s/%s", YARN_MAPPING, yarnVersion, mappingJar);
            String escapedUrl = UrlEscapers.urlFragmentEscaper().escape(mappingJarUrl);

            LOGGER.info("Downloading mapping...");

            File jarFile = new File(STORAGE_DIRECTORY + mappingJar);
            FileUtils.copyURLToFile(URI.create(escapedUrl).toURL(), jarFile);

            try (FileSystem jar = FileSystems.newFileSystem(jarFile.toPath(), (ClassLoader) null))
            {
                Files.copy(jar.getPath(MAPPINGS_PATH), mappingFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            Files.delete(jarFile.toPath());
        }
        return new FileInputStream(mappingFile);
    }

    synchronized public static FileInputStream getCustomMappingStream(String name) throws IOException
    {
        File mappingFile = new File(STORAGE_DIRECTORY + name);
        return new FileInputStream(mappingFile);
    }

    private static String getMappingFileName(String yarnVersion)
    {
        return String.format("yarn-%s-v2", yarnVersion);
    }

    private static String getMappingFileNameFull(String yarnVersion)
    {
        return getMappingFileName(yarnVersion) + ".tiny";
    }

}
